/*****************************************************************************
*  LPC154x  PWM deadtime  program
*
*  Use SCT timer to generate a two-channel double-edge controlled PWM
*  intended for use as a complementary PWM pair with dead-time control
*****************************************************************************/

#include "board.h"


/*****************************************************************************
 * Private types/enumerations/variables
 ****************************************************************************/
#define DC1        (130)                                   			 	// duty cycle 1
#define DC2        (135)                                   			 	// duty cycle 2
#define hperiod    (180)
/*****************************************************************************
 * Public types/enumerations/variables
 ****************************************************************************/

/*****************************************************************************
 * Private functions
 ****************************************************************************/
void SCT1_Init(void);
/*****************************************************************************
 * Public functions
 ****************************************************************************/

void SCT1_Init(void)
{
	Chip_SCT_Init(LPC_SCT1);			                   			 	// enable the SCT1 clock

	Chip_SCT_Config(LPC_SCT1, SCT_CONFIG_16BIT_COUNTER      |			 	// split timers, auto limit
                                  SCT_CONFIG_AUTOLIMIT_L        );

	Chip_SCT_SetControl(LPC_SCT1, SCT_CTRL_BIDIR_L(1));     		 	// configure SCT1 as BIDIR

	Chip_SCT_SetMatchCount(LPC_SCT1, SCT_MATCH_0, hperiod);			 	// match on (half) PWM period
	Chip_SCT_SetMatchReload(LPC_SCT1, SCT_MATCH_0, hperiod);

	Chip_SCT_SetMatchCount(LPC_SCT1, SCT_MATCH_1, DC1);				 	// match on duty cycle 1
	Chip_SCT_SetMatchReload(LPC_SCT1, SCT_MATCH_1, DC1);

	Chip_SCT_SetMatchCount(LPC_SCT1, SCT_MATCH_2, DC2);				 	// match on duty cycle 2
	Chip_SCT_SetMatchReload(LPC_SCT1, SCT_MATCH_2, DC2);

	Chip_SCT_EventState(LPC_SCT1, SCT_EVENT_0, ENABLE_ALL_STATES);	 	// event 0 happens in all states
	Chip_SCT_EventControl(LPC_SCT1, SCT_EVENT_0,(CHIP_SCT_EVENTCTRL_T)  ( SCT_IOCOND_FALL |
                                                                              SCT_COMBMODE_IO ));  	// IN_0 falling edge only condition

	Chip_SCT_EventState(LPC_SCT1, SCT_EVENT_1, ENABLE_ALL_STATES);	 	// event 1 happens in all states
	Chip_SCT_EventControl(LPC_SCT1, SCT_EVENT_1,(CHIP_SCT_EVENTCTRL_T)  ( SCT_IOCOND_RAISE |
                                                                              SCT_COMBMODE_IO  ));  	// IN_0 rising edge only condition

	Chip_SCT_EventState(LPC_SCT1, SCT_EVENT_2, ENABLE_ALL_STATES);	 	// event 2 happens in all states
	Chip_SCT_EventControl(LPC_SCT1, SCT_EVENT_2,(CHIP_SCT_EVENTCTRL_T)  ( SCT_EVECTRL_MATCH1  |
                                                                              SCT_COMBMODE_MATCH  )); 	// match 1 (DC1) only condition

	Chip_SCT_EventState(LPC_SCT1, SCT_EVENT_3, ENABLE_ALL_STATES);	 	// event 3 happens in all states
	Chip_SCT_EventControl(LPC_SCT1, SCT_EVENT_3,(CHIP_SCT_EVENTCTRL_T)  ( SCT_EVECTRL_MATCH2  |
                                                                              SCT_COMBMODE_MATCH  )); 	// match 2 (DC) only condition


	Chip_SCT_SetOutput(LPC_SCT1, SCT_OUTPUT_0, (CHIP_SCT_EVENT_T) ( SCT_EVT_0 |
                                                                        SCT_EVT_2 ));  			// event 0 and 2 set OUT0 (blue LED)

	Chip_SCT_ClearOutput(LPC_SCT1, SCT_OUTPUT_0, SCT_EVT_2);		    // event 2 clears OUT0 (blue LED)

	Chip_SCT_SetOutput(LPC_SCT1, SCT_OUTPUT_1, SCT_EVT_3);			    // event 3 sets OUT1 (red LED)

	Chip_SCT_ClearOutput(LPC_SCT1, SCT_OUTPUT_1, (CHIP_SCT_EVENT_T) ( SCT_EVT_0 |
                                                                          SCT_EVT_3 ));			// event 0 and 3 clear OUT1 (red LED)

	Chip_SCT_SetConflictResolution(LPC_SCT1, 0x00, 0x0F);  				// toggle OUT0 and OUT1 on conflict

	Chip_SCT_Output(LPC_SCT1, 0x01);                   					// default set OUT0 and and clear OUT1

    LPC_SCT1->STOP_L  = SCT_EVE_0 ;                						// event 0 will stop the timer

    Chip_SCT_EnableEventInt(LPC_SCT1,SCT_EVT_1);           				// event 1 will generate an irq

    NVIC_EnableIRQ(SCT1_IRQn);                             				// enable SCT1 interrupt

    Chip_SCT_ClearControl(LPC_SCT1,SCT_CTRL_HALT_L);	   				// unhalt it by clearing bit 2 of the CTRL register
}

void SCT1_IRQHandler(void)
{

	Chip_SCT_SetControl(LPC_SCT1, SCT_EVE_3);			// clear the L counter
	Chip_SCT_ClearControl(LPC_SCT1, SCT_EVE_1);         // start the L counter

	Chip_SCT_SetEventFlag(LPC_SCT1, SCT_EVT_1);			// clear event 1 interrupt flag
}


int main(void)
{
	SystemCoreClockUpdate();
	Board_Init();

	Chip_Clock_SetSysClockDiv(6);

	Chip_SCTIPU_Init();												// enable SCTIPU clock

	Chip_Clock_EnablePeriphClock(	(CHIP_SYSCTL_CLOCK_T) ( SYSCTL_CLOCK_SWM 	  | 			// enable SWM clock
                                                                SYSCTL_CLOCK_MUX 	  |			// enable PMUX clock
                                                                SYSCTL_CLOCK_IOCON	));		    //enable IOCON clock


	Chip_SWM_MovablePortPinAssign(SWM_SCT1_OUT0_O , 1, 1);			//SCT1_OUT0 = P1.1 = blue LED
	Chip_SWM_MovablePortPinAssign(SWM_SCT1_OUT1_O , 0, 25);			//SCT1_OUT1 = P0.25  = red   LED

	Chip_SWM_MovablePortPinAssign(SWM_SCT_ABORT0_I , 1, 9);			// P1.9 (SW2) = SCT_ABORT0

    Chip_SCTIPU_AbortInputEnable( 1, 1);
    Chip_INMUX_SelectSCT1Src(0, SCT1_INMUX_SCTIPU_ABORT);           // SCT1_IN0 = SCTIPU_ABORT = P1.9 (SW2)


    SCT1_Init();                                          			// Initialize SCT0


    while (1)                                              			// loop forever
    {
    	__WFI();
    }
}
